<?php
// api/v1/index.php - API V1 Direct Endpoints

// Set headers
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Handle OPTIONS request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Include required files
require_once '../../core/Database.php';
require_once '../../app/Models/Agent.php';
require_once '../../app/Models/User.php';
require_once '../../app/Models/Game.php';
require_once '../../app/Models/Transaction.php';

// Parse request
$requestUri = $_SERVER['REQUEST_URI'];
$path = parse_url($requestUri, PHP_URL_PATH);
$segments = explode('/', trim($path, '/'));

// Get endpoint
$endpoint = end($segments);

// Initialize models
$agentModel = new Agent();
$userModel = new User();
$gameModel = new Game();
$transactionModel = new Transaction();

// Route to appropriate handler
try {
    switch ($endpoint) {
        case 'connection':
        case 'connection.do.php':
            echo json_encode([
                'status' => 'success',
                'message' => 'API is online',
                'timestamp' => date('Y-m-d H:i:s')
            ]);
            break;
            
        case 'CreateMember.aspx':
            handleCreateMember();
            break;
            
        case 'GetBalance.ashx':
            handleGetBalance();
            break;
            
        case 'GetBalanceAgent.ashx':
            handleGetAgentBalance();
            break;
            
        case 'GetProviderList.aspx':
            handleGetProviderList();
            break;
            
        case 'GetGameList.aspx':
            handleGetGameList();
            break;
            
        case 'LaunchGame.aspx':
            handleLaunchGame();
            break;
            
        case 'MakeTransfer.aspx':
            handleMakeTransfer();
            break;
            
        case 'GetHistoryLog.aspx':
            handleGetHistoryLog();
            break;
            
        default:
            http_response_code(404);
            echo json_encode([
                'status' => 'fail',
                'message' => 'Endpoint not found'
            ]);
            break;
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'status' => 'fail',
        'message' => 'Internal server error'
    ]);
}

// Handler functions
function handleCreateMember() {
    global $agentModel, $userModel;
    
    $agentToken = $_GET['agent_token'] ?? '';
    $agentCode = $_GET['agent_code'] ?? '';
    $username = $_GET['username'] ?? '';
    
    if (empty($agentToken) || empty($agentCode) || empty($username)) {
        echo json_encode(['status' => 'fail', 'message' => 'Invalid parameters']);
        return;
    }
    
    // Validate agent
    $agent = $agentModel->findBy('agent_code', $agentCode);
    if (!$agent || $agent['agent_token'] !== $agentToken) {
        echo json_encode(['status' => 'fail', 'message' => 'Invalid agent credentials']);
        return;
    }
    
    // Check if user exists
    $existingUser = $userModel->findBy('extUser', $username);
    if ($existingUser && $existingUser['agent_code'] === $agentCode) {
        echo json_encode(['status' => 'fail', 'message' => 'Username already exists']);
        return;
    }
    
    // Create user
    $userData = [
        'intUser' => generateRandomString(8),
        'extUser' => $username,
        'password' => generateRandomString(8),
        'agent_code' => $agentCode
    ];
    
    if ($userModel->createUser($userData)) {
        echo json_encode([
            'status' => 'success',
            'username' => $username,
            'message' => 'User created successfully'
        ]);
    } else {
        echo json_encode(['status' => 'fail', 'message' => 'Failed to create user']);
    }
}

function handleGetBalance() {
    global $agentModel, $userModel;
    
    $agentToken = $_GET['agent_token'] ?? '';
    $agentCode = $_GET['agent_code'] ?? '';
    $username = $_GET['username'] ?? '';
    
    if (empty($agentToken) || empty($agentCode) || empty($username)) {
        echo json_encode(['status' => 'fail', 'message' => 'Invalid parameters']);
        return;
    }
    
    // Validate agent
    $agent = $agentModel->findBy('agent_code', $agentCode);
    if (!$agent || $agent['agent_token'] !== $agentToken) {
        echo json_encode(['status' => 'fail', 'message' => 'Invalid agent credentials']);
        return;
    }
    
    // Get user
    $user = $userModel->findBy('extUser', $username);
    if (!$user || $user['agent_code'] !== $agentCode) {
        echo json_encode(['status' => 'fail', 'message' => 'User not found']);
        return;
    }
    
    echo json_encode([
        'status' => 'success',
        'data' => [
            'username' => $username,
            'balance' => (float)$user['balance']
        ],
        'message' => 'Success'
    ]);
}

function handleGetAgentBalance() {
    global $agentModel;
    
    $agentToken = $_GET['agent_token'] ?? '';
    $agentCode = $_GET['agent_code'] ?? '';
    
    if (empty($agentToken) || empty($agentCode)) {
        echo json_encode(['status' => 'fail', 'message' => 'Invalid parameters']);
        return;
    }
    
    // Validate agent
    $agent = $agentModel->findBy('agent_code', $agentCode);
    if (!$agent || $agent['agent_token'] !== $agentToken) {
        echo json_encode(['status' => 'fail', 'message' => 'Invalid agent credentials']);
        return;
    }
    
    echo json_encode([
        'status' => 'success',
        'agent' => [
            'balance' => (float)$agent['agent_balance']
        ],
        'message' => 'Success'
    ]);
}

function generateRandomString($length = 8) {
    $characters = '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz';
    $randomString = '';
    
    for ($i = 0; $i < $length; $i++) {
        $randomString .= $characters[rand(0, strlen($characters) - 1)];
    }
    
    return $randomString;
}

function handleGetProviderList() {
    global $gameModel, $agentModel;
    
    $agentToken = $_GET['agent_token'] ?? '';
    $agentCode = $_GET['agent_code'] ?? '';
    
    if (empty($agentToken) || empty($agentCode)) {
        echo json_encode(['status' => 'fail', 'message' => 'Invalid parameters']);
        return;
    }
    
    $agent = $agentModel->findBy('agent_code', $agentCode);
    if (!$agent || $agent['agent_token'] !== $agentToken) {
        echo json_encode(['status' => 'fail', 'message' => 'Invalid agent credentials']);
        return;
    }
    
    $providers = $gameModel->getProviders();
    $providerData = [];
    
    foreach ($providers as $provider) {
        $providerData[] = [
            'provider_code' => $provider['game_provider'],
            'provider_name' => ucwords(str_replace('_', ' ', $provider['game_provider'])),
            'provider_type' => 'slot',
            'provider_image' => '',
            'provider_status' => 1
        ];
    }
    
    echo json_encode([
        'status' => 'success',
        'provider' => $providerData,
        'message' => 'SUCCESS'
    ]);
}

function handleGetGameList() {
    global $gameModel, $agentModel;
    
    $agentToken = $_GET['agent_token'] ?? '';
    $agentCode = $_GET['agent_code'] ?? '';
    
    if (empty($agentToken) || empty($agentCode)) {
        echo json_encode(['status' => 'fail', 'message' => 'Invalid parameters']);
        return;
    }
    
    $agent = $agentModel->findBy('agent_code', $agentCode);
    if (!$agent || $agent['agent_token'] !== $agentToken) {
        echo json_encode(['status' => 'fail', 'message' => 'Invalid agent credentials']);
        return;
    }
    
    $games = $gameModel->getActiveGames();
    $gameData = [];
    
    foreach ($games as $game) {
        $gameData[] = [
            'game_code' => $game['game_code'],
            'game_name' => $game['game_name'],
            'game_provider' => $game['game_provider'],
            'game_type' => $game['game_type'],
            'game_image' => $game['game_image'],
            'game_status' => (int)$game['game_status']
        ];
    }
    
    echo json_encode([
        'status' => 'success',
        'games' => $gameData,
        'message' => 'SUCCESS'
    ]);
}

function handleLaunchGame() {
    global $gameModel, $userModel, $agentModel;
    
    $agentToken = $_GET['agent_token'] ?? '';
    $agentCode = $_GET['agent_code'] ?? '';
    $username = $_GET['username'] ?? '';
    $gameCode = $_GET['game_code'] ?? '';
    
    if (empty($agentToken) || empty($agentCode) || empty($username) || empty($gameCode)) {
        echo json_encode(['status' => 'fail', 'message' => 'Invalid parameters']);
        return;
    }
    
    $agent = $agentModel->findBy('agent_code', $agentCode);
    if (!$agent || $agent['agent_token'] !== $agentToken) {
        echo json_encode(['status' => 'fail', 'message' => 'Invalid agent credentials']);
        return;
    }
    
    $user = $userModel->findBy('extUser', $username);
    if (!$user || $user['agent_code'] !== $agentCode) {
        echo json_encode(['status' => 'fail', 'message' => 'User not found']);
        return;
    }
    
    $game = $gameModel->findBy('game_code', $gameCode);
    if (!$game || $game['game_status'] != 1) {
        echo json_encode(['status' => 'fail', 'message' => 'Game not available']);
        return;
    }
    
    $sessionId = md5($gameCode . $user['intUser'] . time());
    $gameUrl = "https://playgame.example.com/?session=" . $sessionId;
    
    echo json_encode([
        'status' => 'success',
        'gameUrl' => $gameUrl,
        'message' => 'SUCCESS'
    ]);
}

function handleMakeTransfer() {
    global $agentModel, $userModel, $transactionModel;
    
    $agentToken = $_GET['agent_token'] ?? '';
    $agentCode = $_GET['agent_code'] ?? '';
    $username = $_GET['username'] ?? '';
    $amount = (float)($_GET['amount'] ?? 0);
    $type = $_GET['type'] ?? '';
    
    if (empty($agentToken) || empty($agentCode) || empty($username) || $amount <= 0 || empty($type)) {
        echo json_encode(['status' => 'fail', 'message' => 'Invalid parameters']);
        return;
    }
    
    $agent = $agentModel->findBy('agent_code', $agentCode);
    if (!$agent || $agent['agent_token'] !== $agentToken) {
        echo json_encode(['status' => 'fail', 'message' => 'Invalid agent credentials']);
        return;
    }
    
    $user = $userModel->findBy('extUser', $username);
    if (!$user || $user['agent_code'] !== $agentCode) {
        echo json_encode(['status' => 'fail', 'message' => 'User not found']);
        return;
    }
    
    $beforeBalance = $agent['agent_balance'];
    
    if ($type === 'deposit') {
        if ($beforeBalance < $amount) {
            echo json_encode(['status' => 'fail', 'message' => 'Insufficient agent balance']);
            return;
        }
        
        $agentModel->subtractBalance($agentCode, $amount);
        $userModel->addBalance($username, $amount, $agentCode);
        $afterBalance = $beforeBalance - $amount;
    } else if ($type === 'withdraw') {
        if ($user['balance'] < $amount) {
            echo json_encode(['status' => 'fail', 'message' => 'Insufficient user balance']);
            return;
        }
        
        $userModel->subtractBalance($username, $amount, $agentCode);
        $agentModel->addBalance($agentCode, $amount);
        $afterBalance = $beforeBalance + $amount;
    } else {
        echo json_encode(['status' => 'fail', 'message' => 'Invalid transaction type']);
        return;
    }
    
    echo json_encode([
        'status' => 'success',
        'data' => [
            'agent_code' => $agentCode,
            'action' => strtoupper($type),
            'amount' => $amount,
            'before_balance' => $beforeBalance,
            'after_balance' => $afterBalance
        ],
        'message' => 'SUCCESS'
    ]);
}

function handleGetHistoryLog() {
    global $agentModel, $userModel;
    
    $agentToken = $_GET['agent_token'] ?? '';
    $agentCode = $_GET['agent_code'] ?? '';
    $username = $_GET['username'] ?? '';
    $type = $_GET['type'] ?? '';
    
    if (empty($agentToken) || empty($agentCode) || empty($username) || empty($type)) {
        echo json_encode(['status' => 'fail', 'message' => 'Invalid parameters']);
        return;
    }
    
    $agent = $agentModel->findBy('agent_code', $agentCode);
    if (!$agent || $agent['agent_token'] !== $agentToken) {
        echo json_encode(['status' => 'fail', 'message' => 'Invalid agent credentials']);
        return;
    }
    
    $user = $userModel->findBy('extUser', $username);
    if (!$user || $user['agent_code'] !== $agentCode) {
        echo json_encode(['status' => 'fail', 'message' => 'User not found']);
        return;
    }
    
    // Get history from database
    $db = Database::getInstance()->getConnection();
    $stmt = $db->prepare("
        SELECT * FROM tb_history 
        WHERE extUser = ? AND agent_code = ? 
        AND DATE(created_at) = CURDATE()
        ORDER BY created_at DESC
    ");
    $stmt->execute([$username, $agentCode]);
    $history = $stmt->fetchAll();
    
    $gameLog = [];
    foreach ($history as $record) {
        $gameLog[] = [
            'username' => $record['extUser'],
            'bet_money' => (float)$record['bet_money'],
            'win_money' => (float)$record['win_money'],
            'provider_code' => $record['provider_code'],
            'game_code' => $record['game_code'],
            'type' => $record['type'],
            'txn' => [
                'txn_id' => $record['txn_id'],
                'txn_type' => $record['txn_type']
            ],
            'user_start_balance' => (float)$record['user_start_balance'],
            'user_end_balance' => (float)$record['user_end_balance'],
            'created_at' => $record['created_at']
        ];
    }
    
    echo json_encode([
        'status' => 'success',
        'game_log' => $gameLog,
        'message' => 'SUCCESS'
    ]);
}
?>