<?php

require_once '../core/Controller.php';

class ApiController extends Controller
{
    private $agentModel;
    private $userModel;
    private $gameModel;
    private $transactionModel;

    public function __construct()
    {
        parent::__construct();
        $this->agentModel = $this->loadModel('Agent');
        $this->userModel = $this->loadModel('User');
        $this->gameModel = $this->loadModel('Game');
        $this->transactionModel = $this->loadModel('Transaction');
    }

    public function connection()
    {
        return $this->json([
            'status' => 'success',
            'message' => 'API is online',
            'timestamp' => date('Y-m-d H:i:s')
        ]);
    }

    public function createMember()
    {
        $agentToken = $_GET['agent_token'] ?? '';
        $agentCode = $_GET['agent_code'] ?? '';
        $username = $_GET['username'] ?? '';

        if (empty($agentToken) || empty($agentCode) || empty($username)) {
            return $this->json([
                'status' => 'fail',
                'message' => 'Invalid parameters'
            ]);
        }

        $agent = $this->agentModel->authenticate($agentCode, $agentToken);
        if (!$agent) {
            return $this->json([
                'status' => 'fail',
                'message' => 'Invalid agent credentials'
            ]);
        }

        if ($agent['agent_status'] === 'blocked') {
            return $this->json([
                'status' => 'fail',
                'message' => 'Agent is blocked'
            ]);
        }

        $existingUser = $this->userModel->findBy('extUser', $username);
        if ($existingUser) {
            return $this->json([
                'status' => 'fail',
                'message' => 'Username already exists'
            ]);
        }

        $userData = [
            'intUser' => $this->generateRandomString(8),
            'extUser' => $username,
            'password' => $this->generateRandomString(8),
            'agent_code' => $agentCode
        ];

        if ($this->userModel->createUser($userData)) {
            return $this->json([
                'status' => 'success',
                'username' => $username,
                'message' => 'User created successfully'
            ]);
        }

        return $this->json([
            'status' => 'fail',
            'message' => 'Failed to create user'
        ]);
    }

    public function getBalance()
    {
        $agentToken = $_GET['agent_token'] ?? '';
        $agentCode = $_GET['agent_code'] ?? '';
        $username = $_GET['username'] ?? '';

        if (empty($agentToken) || empty($agentCode) || empty($username)) {
            return $this->json([
                'status' => 'fail',
                'message' => 'Invalid parameters'
            ]);
        }

        $agent = $this->agentModel->findBy('agent_token', $agentToken);
        if (!$agent || $agent['agent_code'] !== $agentCode) {
            return $this->json([
                'status' => 'fail',
                'message' => 'Invalid agent credentials'
            ]);
        }

        $user = $this->userModel->findBy('extUser', $username);
        if (!$user || $user['agent_code'] !== $agentCode) {
            return $this->json([
                'status' => 'fail',
                'message' => 'User not found'
            ]);
        }

        return $this->json([
            'status' => 'success',
            'data' => [
                'username' => $username,
                'balance' => (float)$user['balance']
            ],
            'message' => 'Success'
        ]);
    }

    public function getAgentBalance()
    {
        $agentToken = $_GET['agent_token'] ?? '';
        $agentCode = $_GET['agent_code'] ?? '';

        if (empty($agentToken) || empty($agentCode)) {
            return $this->json([
                'status' => 'fail',
                'message' => 'Invalid parameters'
            ]);
        }

        $agent = $this->agentModel->findBy('agent_token', $agentToken);
        if (!$agent || $agent['agent_code'] !== $agentCode) {
            return $this->json([
                'status' => 'fail',
                'message' => 'Invalid agent credentials'
            ]);
        }

        return $this->json([
            'status' => 'success',
            'agent' => [
                'balance' => (float)$agent['agent_balance']
            ],
            'message' => 'Success'
        ]);
    }

    public function getProviderList()
    {
        $agentToken = $_GET['agent_token'] ?? '';
        $agentCode = $_GET['agent_code'] ?? '';

        if (empty($agentToken) || empty($agentCode)) {
            return $this->json([
                'status' => 'fail',
                'message' => 'Invalid parameters'
            ]);
        }

        $agent = $this->agentModel->findBy('agent_token', $agentToken);
        if (!$agent || $agent['agent_code'] !== $agentCode) {
            return $this->json([
                'status' => 'fail',
                'message' => 'Invalid agent credentials'
            ]);
        }

        $providers = $this->gameModel->getProviders();
        $providerData = [];

        foreach ($providers as $provider) {
            $providerData[] = [
                'provider_code' => $provider['game_provider'],
                'provider_name' => ucwords(str_replace('_', ' ', $provider['game_provider'])),
                'provider_type' => 'slot',
                'provider_image' => '',
                'provider_status' => 1
            ];
        }

        return $this->json([
            'status' => 'success',
            'provider' => $providerData,
            'message' => 'Success'
        ]);
    }

    public function getGameList()
    {
        $agentToken = $_GET['agent_token'] ?? '';
        $agentCode = $_GET['agent_code'] ?? '';

        if (empty($agentToken) || empty($agentCode)) {
            return $this->json([
                'status' => 'fail',
                'message' => 'Invalid parameters'
            ]);
        }

        $agent = $this->agentModel->findBy('agent_token', $agentToken);
        if (!$agent || $agent['agent_code'] !== $agentCode) {
            return $this->json([
                'status' => 'fail',
                'message' => 'Invalid agent credentials'
            ]);
        }

        $games = $this->gameModel->getActiveGames();
        $gameData = [];

        foreach ($games as $game) {
            $gameData[] = [
                'game_code' => $game['game_code'],
                'game_name' => $game['game_name'],
                'game_provider' => $game['game_provider'],
                'game_type' => $game['game_type'],
                'game_image' => $game['game_image'],
                'game_status' => (int)$game['game_status']
            ];
        }

        return $this->json([
            'status' => 'success',
            'games' => $gameData,
            'message' => 'Success'
        ]);
    }

    public function launchGame()
    {
        $agentToken = $_GET['agent_token'] ?? '';
        $agentCode = $_GET['agent_code'] ?? '';
        $username = $_GET['username'] ?? '';
        $gameCode = $_GET['game_code'] ?? '';

        if (empty($agentToken) || empty($agentCode) || empty($username) || empty($gameCode)) {
            return $this->json([
                'status' => 'fail',
                'message' => 'Invalid parameters'
            ]);
        }

        $agent = $this->agentModel->findBy('agent_token', $agentToken);
        if (!$agent || $agent['agent_code'] !== $agentCode) {
            return $this->json([
                'status' => 'fail',
                'message' => 'Invalid agent credentials'
            ]);
        }

        $user = $this->userModel->findBy('extUser', $username);
        if (!$user || $user['agent_code'] !== $agentCode) {
            return $this->json([
                'status' => 'fail',
                'message' => 'User not found'
            ]);
        }

        $game = $this->gameModel->findBy('game_code', $gameCode);
        if (!$game) {
            return $this->json([
                'status' => 'fail',
                'message' => 'Game not found'
            ]);
        }

        if ($game['game_status'] != 1) {
            return $this->json([
                'status' => 'fail',
                'message' => 'Game is under maintenance'
            ]);
        }

        // Generate game session
        $sessionId = md5($gameCode . $user['intUser'] . time());
        $gameUrl = "https://playgame.example.com/?session=" . $sessionId;

        return $this->json([
            'status' => 'success',
            'gameUrl' => $gameUrl,
            'message' => 'Success'
        ]);
    }

    public function makeTransfer()
    {
        $agentToken = $_GET['agent_token'] ?? '';
        $agentCode = $_GET['agent_code'] ?? '';
        $username = $_GET['username'] ?? '';
        $amount = (float)($_GET['amount'] ?? 0);
        $type = $_GET['type'] ?? '';

        if (empty($agentToken) || empty($agentCode) || empty($username) || $amount <= 0 || empty($type)) {
            return $this->json([
                'status' => 'fail',
                'message' => 'Invalid parameters'
            ]);
        }

        $agent = $this->agentModel->findBy('agent_token', $agentToken);
        if (!$agent || $agent['agent_code'] !== $agentCode) {
            return $this->json([
                'status' => 'fail',
                'message' => 'Invalid agent credentials'
            ]);
        }

        $user = $this->userModel->findBy('extUser', $username);
        if (!$user || $user['agent_code'] !== $agentCode) {
            return $this->json([
                'status' => 'fail',
                'message' => 'User not found'
            ]);
        }

        $beforeBalance = $agent['agent_balance'];

        if ($type === 'deposit') {
            if ($beforeBalance < $amount) {
                return $this->json([
                    'status' => 'fail',
                    'message' => 'Insufficient agent balance'
                ]);
            }

            $this->agentModel->subtractBalance($agentCode, $amount);
            $this->userModel->addBalance($username, $amount, $agentCode);
            $afterBalance = $beforeBalance - $amount;
        } else if ($type === 'withdraw') {
            if ($user['balance'] < $amount) {
                return $this->json([
                    'status' => 'fail',
                    'message' => 'Insufficient user balance'
                ]);
            }

            $this->userModel->subtractBalance($username, $amount, $agentCode);
            $this->agentModel->addBalance($agentCode, $amount);
            $afterBalance = $beforeBalance + $amount;
        } else {
            return $this->json([
                'status' => 'fail',
                'message' => 'Invalid transaction type'
            ]);
        }

        // Record transaction
        $this->transactionModel->createTransaction([
            'trxid' => md5(uniqid()),
            'intUser' => $user['intUser'],
            'extUser' => $username,
            'type' => $type,
            'amount' => $amount,
            'before_agent_balance' => $beforeBalance,
            'after_agent_balance' => $afterBalance,
            'agent_code' => $agentCode
        ]);

        return $this->json([
            'status' => 'success',
            'data' => [
                'agent_code' => $agentCode,
                'action' => strtoupper($type),
                'amount' => $amount,
                'before_balance' => $beforeBalance,
                'after_balance' => $afterBalance
            ],
            'message' => 'Success'
        ]);
    }

    private function generateRandomString($length = 8)
    {
        $characters = '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz';
        $randomString = '';
        
        for ($i = 0; $i < $length; $i++) {
            $randomString .= $characters[rand(0, strlen($characters) - 1)];
        }
        
        return $randomString;
    }
}