<?php
// app/Controllers/ApiV1Controller.php - API V1 Controller (Complete)

require_once '../core/Controller.php';

class ApiV1Controller extends Controller
{
    private $agentModel;
    private $userModel;
    private $gameModel;
    private $transactionModel;

    public function __construct()
    {
        parent::__construct();
        header('Content-Type: application/json');
        $this->agentModel = $this->loadModel('Agent');
        $this->userModel = $this->loadModel('User');
        $this->gameModel = $this->loadModel('Game');
        $this->transactionModel = $this->loadModel('Transaction');
    }

    // Test connection endpoint
    public function connection()
    {
        return $this->json([
            'status' => 'success',
            'message' => 'API is online',
            'timestamp' => date('Y-m-d H:i:s'),
            'version' => '1.0.0'
        ]);
    }

    // Create new member/user
    public function createMember()
    {
        $params = $this->validateApiParams(['agent_token', 'agent_code', 'username']);
        if ($params['error']) {
            return $this->json($params['response'], 400);
        }

        $agent = $this->validateAgent($params['agent_token'], $params['agent_code']);
        if (!$agent['valid']) {
            return $this->json($agent['response'], 401);
        }

        // Check if user already exists
        $existingUser = $this->userModel->findBy('extUser', $params['username']);
        if ($existingUser && $existingUser['agent_code'] === $params['agent_code']) {
            return $this->json([
                'status' => 'fail',
                'message' => 'Username already exists'
            ], 409);
        }

        // Create new user
        $userData = [
            'intUser' => $this->generateRandomString(8),
            'extUser' => $params['username'],
            'password' => $this->generateRandomString(8),
            'agent_code' => $params['agent_code']
        ];

        if ($this->userModel->createUser($userData)) {
            return $this->json([
                'status' => 'success',
                'username' => $params['username'],
                'message' => 'User created successfully'
            ]);
        }

        return $this->json([
            'status' => 'fail',
            'message' => 'Failed to create user'
        ], 500);
    }

    // Get user balance
    public function getBalance()
    {
        $params = $this->validateApiParams(['agent_token', 'agent_code', 'username']);
        if ($params['error']) {
            return $this->json($params['response'], 400);
        }

        $agent = $this->validateAgent($params['agent_token'], $params['agent_code']);
        if (!$agent['valid']) {
            return $this->json($agent['response'], 401);
        }

        $user = $this->userModel->findBy('extUser', $params['username']);
        if (!$user || $user['agent_code'] !== $params['agent_code']) {
            return $this->json([
                'status' => 'fail',
                'message' => 'User not found'
            ], 404);
        }

        return $this->json([
            'status' => 'success',
            'data' => [
                'username' => $params['username'],
                'balance' => (float)$user['balance']
            ],
            'message' => 'Success'
        ]);
    }

    // Get agent balance
    public function getAgentBalance()
    {
        $params = $this->validateApiParams(['agent_token', 'agent_code']);
        if ($params['error']) {
            return $this->json($params['response'], 400);
        }

        $agent = $this->validateAgent($params['agent_token'], $params['agent_code']);
        if (!$agent['valid']) {
            return $this->json($agent['response'], 401);
        }

        return $this->json([
            'status' => 'success',
            'agent' => [
                'balance' => (float)$agent['data']['agent_balance']
            ],
            'message' => 'Success'
        ]);
    }

    // Get game providers
    public function getProviderList()
    {
        $params = $this->validateApiParams(['agent_token', 'agent_code']);
        if ($params['error']) {
            return $this->json($params['response'], 400);
        }

        $agent = $this->validateAgent($params['agent_token'], $params['agent_code']);
        if (!$agent['valid']) {
            return $this->json($agent['response'], 401);
        }

        $providers = $this->gameModel->getProviders();
        $providerData = [];

        foreach ($providers as $provider) {
            $providerData[] = [
                'provider_code' => $provider['game_provider'],
                'provider_name' => ucwords(str_replace('_', ' ', $provider['game_provider'])),
                'provider_type' => 'slot',
                'provider_image' => '',
                'provider_status' => 1
            ];
        }

        return $this->json([
            'status' => 'success',
            'provider' => $providerData,
            'message' => 'Success'
        ]);
    }

    // Get game list
    public function getGameList()
    {
        $params = $this->validateApiParams(['agent_token', 'agent_code']);
        if ($params['error']) {
            return $this->json($params['response'], 400);
        }

        $agent = $this->validateAgent($params['agent_token'], $params['agent_code']);
        if (!$agent['valid']) {
            return $this->json($agent['response'], 401);
        }

        $games = $this->gameModel->getActiveGames();
        $gameData = [];

        foreach ($games as $game) {
            $gameData[] = [
                'game_code' => $game['game_code'],
                'game_name' => $game['game_name'],
                'game_provider' => $game['game_provider'],
                'game_type' => $game['game_type'],
                'game_image' => $game['game_image'],
                'game_status' => (int)$game['game_status']
            ];
        }

        return $this->json([
            'status' => 'success',
            'games' => $gameData,
            'message' => 'Success'
        ]);
    }

    // Launch game
    public function launchGame()
    {
        $params = $this->validateApiParams(['agent_token', 'agent_code', 'username', 'game_code']);
        if ($params['error']) {
            return $this->json($params['response'], 400);
        }

        $agent = $this->validateAgent($params['agent_token'], $params['agent_code']);
        if (!$agent['valid']) {
            return $this->json($agent['response'], 401);
        }

        $user = $this->userModel->findBy('extUser', $params['username']);
        if (!$user || $user['agent_code'] !== $params['agent_code']) {
            return $this->json([
                'status' => 'fail',
                'message' => 'User not found'
            ], 404);
        }

        $game = $this->gameModel->findBy('game_code', $params['game_code']);
        if (!$game) {
            return $this->json([
                'status' => 'fail',
                'message' => 'Game not found'
            ], 404);
        }

        if ($game['game_status'] != 1) {
            return $this->json([
                'status' => 'fail',
                'message' => 'Game is under maintenance'
            ], 503);
        }

        // Launch game through Nexus Gaming API
        require_once '../app/Services/NexusGamingApi.php';
        $api = new NexusGamingApi($params['agent_code'], $params['agent_token']);
        
        $launchResponse = $api->launchGame($user['intUser'], $game['game_provider'], $params['game_code']);
        
        if ($launchResponse['status'] == 1) {
            // Store game session for tracking
            $sessionId = md5($params['game_code'] . $user['intUser'] . time());
            $stmt = $this->db->prepare("
                INSERT INTO tb_play (extUser, intUser, provider, gamecode, tc, gamestart, url, title) 
                VALUES (?, ?, ?, ?, ?, NOW(), ?, ?)
            ");
            $stmt->execute([
                $params['username'],
                $user['intUser'],
                $game['game_provider'],
                $params['game_code'],
                $sessionId,
                $launchResponse['launch_url'],
                $game['game_name']
            ]);
            
            return $this->json([
                'status' => 'success',
                'gameUrl' => $launchResponse['launch_url'],
                'message' => 'Success'
            ]);
        } else {
            return $this->json([
                'status' => 'fail',
                'message' => $launchResponse['msg']
            ], 400);
        }
    }

    // Make transfer (deposit/withdraw)
    public function makeTransfer()
    {
        $params = $this->validateApiParams(['agent_token', 'agent_code', 'username', 'amount', 'type']);
        if ($params['error']) {
            return $this->json($params['response'], 400);
        }

        $agent = $this->validateAgent($params['agent_token'], $params['agent_code']);
        if (!$agent['valid']) {
            return $this->json($agent['response'], 401);
        }

        $amount = (float)$params['amount'];
        if ($amount <= 0) {
            return $this->json([
                'status' => 'fail',
                'message' => 'Invalid amount'
            ], 400);
        }

        $user = $this->userModel->findBy('extUser', $params['username']);
        if (!$user || $user['agent_code'] !== $params['agent_code']) {
            return $this->json([
                'status' => 'fail',
                'message' => 'User not found'
            ], 404);
        }

        // Use Nexus Gaming API for transactions
        require_once '../app/Services/NexusGamingApi.php';
        $api = new NexusGamingApi($params['agent_code'], $params['agent_token']);
        
        $agentSign = 'TXN_' . time() . '_' . rand(1000, 9999);
        
        if ($params['type'] === 'deposit') {
            $response = $api->userDeposit($user['intUser'], $amount, $agentSign);
        } else if ($params['type'] === 'withdraw') {
            $response = $api->userWithdraw($user['intUser'], $amount, $agentSign);
        } else {
            return $this->json([
                'status' => 'fail',
                'message' => 'Invalid transaction type'
            ], 400);
        }
        
        if ($response['status'] != 1) {
            return $this->json([
                'status' => 'fail',
                'message' => $response['msg']
            ], 400);
        }
        
        // Update local balances based on API response
        $this->agentModel->updateBalance($params['agent_code'], $response['agent_balance']);
        $this->userModel->updateBalance($params['username'], $response['user_balance'], $params['agent_code']);
        
        $beforeBalance = $agent['data']['agent_balance'];
        $afterBalance = $response['agent_balance'];

        // Record transaction
        $this->transactionModel->createTransaction([
            'trxid' => md5(uniqid()),
            'intUser' => $user['intUser'],
            'extUser' => $params['username'],
            'type' => $params['type'],
            'amount' => $amount,
            'before_agent_balance' => $beforeBalance,
            'after_agent_balance' => $afterBalance,
            'agent_code' => $params['agent_code']
        ]);

        return $this->json([
            'status' => 'success',
            'data' => [
                'agent_code' => $params['agent_code'],
                'action' => strtoupper($params['type']),
                'amount' => $amount,
                'before_balance' => $beforeBalance,
                'after_balance' => $afterBalance
            ],
            'message' => 'Success'
        ]);
    }

    // Get history log
    public function getHistoryLog()
    {
        $params = $this->validateApiParams(['agent_token', 'agent_code', 'username', 'type']);
        if ($params['error']) {
            return $this->json($params['response'], 400);
        }

        $agent = $this->validateAgent($params['agent_token'], $params['agent_code']);
        if (!$agent['valid']) {
            return $this->json($agent['response'], 401);
        }

        $user = $this->userModel->findBy('extUser', $params['username']);
        if (!$user || $user['agent_code'] !== $params['agent_code']) {
            return $this->json([
                'status' => 'fail',
                'message' => 'User not found'
            ], 404);
        }

        // Get real history from Nexus Gaming API
        require_once '../app/Services/NexusGamingApi.php';
        $api = new NexusGamingApi($params['agent_code'], $params['agent_token']);
        
        $startDate = date('Y-m-d') . ' 00:00:00';
        $endDate = date('Y-m-d H:i:s');
        
        $historyResponse = $api->getGameLog($user['intUser'], $params['type'], $startDate, $endDate);
        $historyData = [];
        
        if ($historyResponse['status'] == 1 && isset($historyResponse[$params['type']])) {
            foreach ($historyResponse[$params['type']] as $record) {
                $historyData[] = [
                    'username' => $params['username'],
                    'bet_money' => (float)$record['bet_money'],
                    'win_money' => (float)$record['win_money'],
                    'provider_code' => $record['provider_code'],
                    'game_code' => $record['game_code'],
                    'type' => $record['type'],
                    'txn' => [
                        'txn_id' => $record['txn_id'],
                        'txn_type' => $record['txn_type']
                    ],
                    'user_start_balance' => (float)$record['user_start_balance'],
                    'user_end_balance' => (float)$record['user_end_balance'],
                    'created_at' => $record['created_at']
                ];
            }
        }

        return $this->json([
            'status' => 'success',
            'game_log' => $historyData,
            'message' => 'Success'
        ]);
    }

    // Helper methods
    private function validateApiParams($requiredParams)
    {
        $params = [];
        $missing = [];

        foreach ($requiredParams as $param) {
            $value = $_GET[$param] ?? $_POST[$param] ?? '';
            if (empty($value)) {
                $missing[] = $param;
            } else {
                $params[$param] = $value;
            }
        }

        if (!empty($missing)) {
            return [
                'error' => true,
                'response' => [
                    'status' => 'fail',
                    'message' => 'Missing required parameters: ' . implode(', ', $missing)
                ]
            ];
        }

        return array_merge(['error' => false], $params);
    }

    private function validateAgent($token, $code)
    {
        $agent = $this->agentModel->findBy('agent_code', $code);
        
        if (!$agent || $agent['agent_token'] !== $token) {
            return [
                'valid' => false,
                'response' => [
                    'status' => 'fail',
                    'message' => 'Invalid agent credentials'
                ]
            ];
        }

        if ($agent['agent_status'] === 'blocked') {
            return [
                'valid' => false,
                'response' => [
                    'status' => 'fail',
                    'message' => 'Agent account is blocked'
                ]
            ];
        }

        return [
            'valid' => true,
            'data' => $agent
        ];
    }

    private function generateRandomString($length = 8)
    {
        $characters = '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz';
        $randomString = '';
        
        for ($i = 0; $i < $length; $i++) {
            $randomString .= $characters[rand(0, strlen($characters) - 1)];
        }
        
        return $randomString;
    }
}