<?php

class Router
{
    private $routes = [];
    private $middlewares = [];
    private $currentGroup = '';
    private $currentMiddlewares = [];

    public function get($uri, $controller, $middlewares = [])
    {
        $this->addRoute('GET', $uri, $controller, $middlewares);
    }

    public function post($uri, $controller, $middlewares = [])
    {
        $this->addRoute('POST', $uri, $controller, $middlewares);
    }

    public function put($uri, $controller, $middlewares = [])
    {
        $this->addRoute('PUT', $uri, $controller, $middlewares);
    }

    public function delete($uri, $controller, $middlewares = [])
    {
        $this->addRoute('DELETE', $uri, $controller, $middlewares);
    }

    public function group($prefix, $middlewares, $callback)
    {
        $previousGroup = $this->currentGroup;
        $previousMiddlewares = $this->currentMiddlewares;
        
        $this->currentGroup = trim($previousGroup . '/' . $prefix, '/');
        $this->currentMiddlewares = array_merge($previousMiddlewares, $middlewares);
        
        call_user_func($callback, $this);
        
        $this->currentGroup = $previousGroup;
        $this->currentMiddlewares = $previousMiddlewares;
    }

    private function addRoute($method, $uri, $controller, $middlewares = [])
    {
        $fullUri = trim($this->currentGroup . '/' . $uri, '/');
        $allMiddlewares = array_merge($this->currentMiddlewares, $middlewares);
        
        $this->routes[$method][$fullUri] = [
            'controller' => $controller,
            'middlewares' => $allMiddlewares,
            'pattern' => $this->createPattern($fullUri)
        ];
    }

    private function createPattern($uri)
    {
        // Convert {param} to regex patterns
        $pattern = preg_replace('/\{([a-zA-Z0-9_]+)\}/', '(?P<$1>[^/]+)', $uri);
        return '#^' . $pattern . '$#';
    }

    public function route($method, $uri)
    {
        if (!isset($this->routes[$method])) {
            return false;
        }

        foreach ($this->routes[$method] as $routeUri => $route) {
            if (preg_match($route['pattern'], $uri, $matches)) {
                return $this->executeRoute($route, $matches);
            }
        }

        return false;
    }

    private function executeRoute($route, $matches = [])
    {
        // Execute middlewares
        foreach ($route['middlewares'] as $middleware) {
            $middlewareResult = $this->executeMiddleware($middleware);
            if ($middlewareResult !== true) {
                return $middlewareResult;
            }
        }

        // Parse controller and action
        if (is_string($route['controller'])) {
            [$controllerName, $action] = explode('@', $route['controller']);
        } else {
            return call_user_func($route['controller'], $matches);
        }

        // Load and execute controller
        $controllerFile = "../app/Controllers/{$controllerName}.php";
        if (!file_exists($controllerFile)) {
            throw new Exception("Controller {$controllerName} not found");
        }

        require_once $controllerFile;
        
        if (!class_exists($controllerName)) {
            throw new Exception("Controller class {$controllerName} not found");
        }

        $controller = new $controllerName();
        
        if (!method_exists($controller, $action)) {
            throw new Exception("Method {$action} not found in {$controllerName}");
        }

        // Extract parameters from matches
        $params = [];
        foreach ($matches as $key => $value) {
            if (!is_numeric($key)) {
                $params[$key] = $value;
            }
        }

        return $controller->$action($params);
    }

    private function executeMiddleware($middleware)
    {
        if (is_string($middleware)) {
            $middlewareFile = "../app/Middleware/{$middleware}.php";
            if (file_exists($middlewareFile)) {
                require_once $middlewareFile;
                $middlewareInstance = new $middleware();
                return $middlewareInstance->handle();
            }
        } elseif (is_callable($middleware)) {
            return call_user_func($middleware);
        }
        
        return true;
    }
}