<?php
// cron/monitor-system.php - System health monitoring

require_once '../core/Database.php';

// Set timezone
date_default_timezone_set('Asia/Jakarta');

echo "[" . date('Y-m-d H:i:s') . "] Starting system monitoring...\n";

try {
    $db = Database::getInstance()->getConnection();
    $issues = [];
    
    // Check database connection
    if (!$db) {
        $issues[] = "Database connection failed";
    }
    
    // Check disk space
    $freeSpace = disk_free_space('.');
    $totalSpace = disk_total_space('.');
    $usedPercent = (($totalSpace - $freeSpace) / $totalSpace) * 100;
    
    if ($usedPercent > 90) {
        $issues[] = "Disk space usage is critical: " . round($usedPercent, 2) . "%";
    }
    
    // Check log file sizes
    $logDir = '../logs/';
    if (is_dir($logDir)) {
        $logFiles = glob($logDir . '*.log');
        foreach ($logFiles as $logFile) {
            $fileSize = filesize($logFile);
            if ($fileSize > 100 * 1024 * 1024) { // 100MB
                $issues[] = "Large log file detected: " . basename($logFile) . " (" . round($fileSize / 1024 / 1024, 2) . "MB)";
            }
        }
    }
    
    // Check table sizes
    $stmt = $db->query("
        SELECT table_name, ROUND(((data_length + index_length) / 1024 / 1024), 2) AS size_mb
        FROM information_schema.TABLES 
        WHERE table_schema = DATABASE()
        ORDER BY size_mb DESC
    ");
    
    $largeTables = [];
    while ($row = $stmt->fetch()) {
        if ($row['size_mb'] > 1000) { // 1GB
            $largeTables[] = $row['table_name'] . " (" . $row['size_mb'] . "MB)";
        }
    }
    
    if (!empty($largeTables)) {
        $issues[] = "Large database tables detected: " . implode(', ', $largeTables);
    }
    
    // Check for failed API connections (mock)
    $apiStatus = @file_get_contents('http://localhost/api/v1/connection');
    if ($apiStatus === false) {
        $issues[] = "API endpoint not responding";
    }
    
    // Generate report
    $report = [
        'timestamp' => date('Y-m-d H:i:s'),
        'status' => empty($issues) ? 'healthy' : 'warning',
        'disk_usage' => round($usedPercent, 2) . '%',
        'issues' => $issues,
        'uptime' => exec('uptime'),
        'memory_usage' => memory_get_usage(true) / 1024 / 1024 . ' MB'
    ];
    
    // Save report
    file_put_contents('../logs/system-health.json', json_encode($report, JSON_PRETTY_PRINT));
    
    if (empty($issues)) {
        echo "[" . date('Y-m-d H:i:s') . "] System status: HEALTHY\n";
    } else {
        echo "[" . date('Y-m-d H:i:s') . "] System status: WARNING\n";
        foreach ($issues as $issue) {
            echo "[" . date('Y-m-d H:i:s') . "] ISSUE: $issue\n";
        }
        
        // Send alert (implement email/SMS notification here)
        // sendAlert($issues);
    }

} catch (Exception $e) {
    echo "[" . date('Y-m-d H:i:s') . "] ERROR: " . $e->getMessage() . "\n";
    
    // Log to file
    file_put_contents('../logs/monitor-system.log', date('Y-m-d H:i:s') . " ERROR: " . $e->getMessage() . "\n", FILE_APPEND);
}